<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Accion;
use App\Area;
use App\Auditoria;
use App\Cargo;
use App\Departamento;
use App\Regional;
use App\Rol;
use App\User;
use Carbon\Carbon;
use Session;

class UsuariosController extends Controller
{
    public function index() {
        if (Auth::user()->rol_id < 3) {
            $orden = (isset($_GET['orden']) ? substr($_GET['orden'], 0, strpos($_GET['orden'], '|')) : 'users.created_at');
            $sort = (isset($_GET['orden']) ? substr($_GET['orden'], strpos($_GET['orden'], '|') + 1) : 'desc');
            $departamento = (isset($_GET['departamento']) ? $_GET['departamento'] : null);
            $nombre = (isset($_GET['nombre']) ? $_GET['nombre'] : null);

            if (isset($_GET['perPage'])) {
                Session::put('pagination', $_GET['perPage']);
            } elseif (!Session::has('pagination')) {
                Session::put('pagination', 10);
            }

            $departamentos = Departamento::all()->sortBy('descripcion');
            $regionales = Regional::all()->sortBy('regional');
            $usuarios = User::leftJoin('cargos AS car', 'users.cargo_id', 'car.id')
                ->leftJoin('areas AS are', 'car.area_id', 'are.id')
                ->leftJoin('departamentos AS dep', 'are.departamento_id', 'dep.id')
                ->when($departamento, function ($query) use ($departamento) {
                    return $query->where('dep.id', $departamento);
                })
                ->when($nombre, function ($query) use ($nombre) {
                    return $query->whereRaw('UPPER(users.nombre) LIKE ?', '%' . mb_strtoupper(str_replace(' ', '%', $nombre)) . '%')
                                ->orWhereRaw('UPPER(users.documento) LIKE ?', '%' . mb_strtoupper(str_replace(' ', '%', $nombre)) . '%');
                })
                ->orderBy($orden, $sort)
                ->paginate(Session::get('pagination'));
            return view('usuarios.index')->with(compact('usuarios', 'departamentos', 'regionales', 'orden', 'sort', 'departamento', 'nombre'));
        } else {
            abort(403);
        }
    }

    public function create() {
        if (Auth::user()->rol_id < 3) {
            $departamentos = Departamento::all()->sortBy('descripcion');
            $areas = Area::all()->sortBy('descripcion');
            $cargos = Cargo::all()->sortBy('descripcion');
            $regionales = Regional::all()->sortBy('nombre');
            $roles = Rol::all()->sortByDesc('id');
            return view('usuarios.data')->with(compact('departamentos', 'regionales', 'areas', 'cargos', 'roles'));
        } else {
            abort(403);
        }
    }
    
    public function store(Request $request) {
        if (Auth::user()->rol_id >= 3)
            abort(403);
        
        $usuario = new User();
        try {
            // Inicia la tansacción
            DB::beginTransaction();

            $usuario->fill($request->all());
            $usuario->password = Hash::make($request->documento);
            if ($usuario->regional_id == 'null');
                $usuario->regional_id = null;
            $usuario->save();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Usuarios';
            $auditoria->observacion = 'Usuario creado: ' . $usuario->nombre . ' con número de documento ' . $usuario->documento;
            $auditoria->user_id = Auth::id();
            $auditoria->usuario_id = $usuario->id;
            $auditoria->accion_id = 5;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Confirma los cambios realizados en la BD y redirige a la vista del usuario
            DB::commit();
            return redirect()->route('usuarios.ver', ['documento' => $usuario->documento]);
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            dd($exc);
            $success = false;
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function show($documento) {
        if (Auth::user()->rol_id < 3) {
            $hoy = Carbon::now();
            $usuario = User::where('documento', $documento)->firstOrFail();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Usuarios';
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 8;
            $auditoria->usuario_id = $usuario->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();
            return view('usuarios.ver')->with(compact('usuario', 'hoy'));
        } else {
            abort(403);
        }
    }

    public function traceability($documento) {
        if (Auth::user()->rol_id < 3) {
            $hoy = Carbon::now();
            $usuario = User::where('documento', $documento)->firstOrFail();

            $fecha_inicial = (isset($_GET['fecha_inicial']) ? $_GET['fecha_inicial'] : Carbon::now()->firstOfMonth()->startOfDay()->toDateString());
            $fecha_final = (isset($_GET['fecha_final']) ? $_GET['fecha_final'] : $hoy->endOfDay()->toDateString());
            $accion = (isset($_GET['accion']) ? $_GET['accion'] : null);

            if (isset($_GET['perPage'])) {
                Session::put('pagination', $_GET['perPage']);
            } elseif (!Session::has('pagination')) {
                Session::put('pagination', 10);
            }

            $acciones = Accion::all()->sortBy('descripcion');
            $auditorias = Auditoria::where('user_id', $usuario->id)
                ->whereDate('created_at', '>=', $fecha_inicial)
                ->whereDate('created_at', '<=', $fecha_final)
                ->when($accion, function ($query) use ($accion) {
                    return $query->where('accion_id', $accion);
                })
                ->orderBy('created_at', 'desc')
                ->paginate(Session::get('pagination'));
            return view('usuarios.traceability')->with(compact('usuario', 'hoy', 'auditorias', 'acciones', 'fecha_inicial', 'fecha_final', 'accion'));
        } else {
            abort(403);
        }
    }

    public function edit($documento) {
        if (Auth::user()->rol_id < 3) {
            $usuario = User::where('documento', $documento)->firstOrFail();
            $departamentos = Departamento::all()->sortBy('descripcion');
            $areas = Area::all()->sortBy('descripcion');
            $cargos = Cargo::all()->sortBy('descripcion');
            $regionales = Regional::all()->sortBy('nombre');
            $roles = Rol::all()->sortByDesc('id');
            return view('usuarios.data')->with(compact('usuario', 'departamentos', 'regionales', 'areas', 'cargos', 'roles'));
        } else {
            abort(403);
        }
    }

    public function update(Request $request, $documento) {
        if (Auth::user()->rol_id >= 3)
            abort(403);
        
        $usuario = User::where('documento', $documento)->firstOrFail();
        try {
            // Inicia la tansacción
            DB::beginTransaction();

            $usuario->fill($request->all());
            if ($usuario->regional_id == 'null') {
                $usuario->regional_id = null;
            }
            $usuario->save();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Usuarios';
            $auditoria->observacion = 'Usuario modificado: ' . $usuario->nombre . ' con número de documento ' . $usuario->documento;
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 4;
            $auditoria->usuario_id = $usuario->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Confirma los cambios realizados en la BD y redirige a la vista del usuario
            DB::commit();
            return redirect()->route('usuarios.ver', ['documento' => $usuario->documento]);
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            dd($exc);
            $success = false;
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function delete($documento) {
        $hoy = Carbon::now();
        $usuario = User::where('documento', $documento)->firstOrFail();
        if (Auth::user()->rol_id < 3 && Auth::user()->rol_id < $usuario->rol_id) {
            return view('usuarios.eliminar')->with(compact('usuario', 'hoy'));
        } else {
            abort(403);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $documento)
    {
        $usuario = User::where('documento', $documento)->firstOrFail();
        if (Auth::user()->rol_id >= 3 || Auth::user()->rol_id >= $usuario->rol_id)
            abort(403);
        
        try {
            // Inicia la tansacción
            DB::beginTransaction();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Usuarios';
            $auditoria->observacion = 'Usuario eliminado: ' . $usuario->nombre . ' con número de documento ' . $usuario->documento;
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 6;
            $auditoria->usuario_id = $usuario->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Borra al usuario
            $usuario->delete();
            
            // Confirma los cambios realizados en la BD y redirige a la vista del usuario
            DB::commit();
            return redirect()->route('usuarios');
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            dd($exc);
            $success = false;
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function profile() {
        $hoy = Carbon::now();
        $usuario = Auth::user();

        // Guarda la acción en la tabla de auditorias
        $auditoria = new Auditoria();
        $auditoria->modulo = 'Usuarios';
        $auditoria->user_id = Auth::id();
        $auditoria->accion_id = 8;
        $auditoria->usuario_id = $usuario->id;
        $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
        $auditoria->ip_avicampo = \Request::ip();
        $auditoria->save();
        return view('usuarios.ver')->with(compact('usuario', 'hoy'));
    }

    public function validation($documento) {
        if (Auth::user()->rol_id < 3) {
            $usuario = User::where('documento', $documento)->first();
            return response()->json(compact('usuario'), 200);
        } else {
            abort(403);
        }
    }

    public function claves() {
        $claves = [
        ];

        foreach ($claves as $c) {
            echo Hash::make($c) . '<br>';
        }
    }
}
