<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Auditoria;
use App\Cargo;
use App\Categoria;
use App\Departamento;
use App\Documento;
use App\Firma;
use App\Regional;
use App\Version;
use Carbon\Carbon;
use Session;

class DocumentosController extends Controller
{
    // Middleware que valida si el usuario está autenticado antes de acceder a cualquier método
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('password.check');
    }
    
    public function index() {
        $orden = (isset($_GET['orden']) ? substr($_GET['orden'], 0, strpos($_GET['orden'], '|')) : 'doc.created_at');
        $sort = (isset($_GET['orden']) ? substr($_GET['orden'], strpos($_GET['orden'], '|') + 1) : 'desc');
        $tipo = (isset($_GET['tipo']) ? $_GET['tipo'] : null);
        $nombre = (isset($_GET['nombre']) ? $_GET['nombre'] : null);

        if (isset($_GET['perPage'])) {
            Session::put('pagination', $_GET['perPage']);
        } elseif (!Session::has('pagination')) {
            Session::put('pagination', 10);
        }

        $cargo = Auth::user()->cargo->id;
        $regional = Auth::user()->regional_id;
        $categorias = Categoria::all()->sortBy('descripcion');
        $documentos = DB::table('documentos AS doc')
            ->leftJoin('categorias AS cat', 'doc.categoria_id', 'cat.id')
            ->leftJoin('versiones AS ver', function ($join) {
                $join->on('doc.id', 'ver.documento_id')
                ->where('ver.version', DB::raw("(SELECT MAX(`version`) FROM versiones WHERE documento_id = ver.documento_id AND deleted_at IS NULL)"));
            })
            ->select('doc.id', 'doc.codigo', 'doc.descripcion', 'doc.firmar', 'ver.version', 'ver.tipo', 'ver.url', 'cat.descripcion AS categoria', 'ver.created_at', 'doc.updated_at')
            ->whereNull('doc.deleted_at')
            ->when($tipo, function ($query) use ($tipo) {
                return $query->where('doc.categoria_id', $tipo);
            })
            ->when($nombre, function ($query) use ($nombre) {
                return $query->whereRaw('UPPER(doc.descripcion) LIKE ?', '%' . mb_strtoupper(str_replace(' ', '%', $nombre)) . '%')
                            ->orWhereRaw('UPPER(doc.codigo) LIKE ?', '%' . mb_strtoupper(str_replace(' ', '%', $nombre)) . '%');
            })
            ->orderBy($orden, $sort)
            ->paginate(Session::get('pagination'));
        return view('documentos.index')->with(compact('documentos', 'categorias', 'orden', 'sort', 'tipo', 'nombre'));
    }
    
    public function create() {
        $departamentos = Departamento::all()->sortBy('descripcion');
        $categorias = Categoria::all()->sortBy('descripcion');
        $regionales = Regional::all()->sortBy('regional');
        return view('documentos.create')->with(compact('departamentos', 'categorias', 'regionales'));
    }
    
    public function store(Request $request) {
        if (Auth::user()->rol_id >= 4)
            abort(403);
        
        $documento = new Documento();
        $version = new Version();
        try {
            // Inicia la tansacción
            DB::beginTransaction();
            
            $documento->fill($request->all());
            $documento->codigo = mb_strtoupper(trim($documento->codigo));
            $documento->descripcion = mb_strtoupper(trim($documento->descripcion));
            $documento->descargar = true;
            if (isset($request->firmar))
                $documento->firmar = true;
            $documento->save();

            $version->fill($request->all());
            $version->documento_id = $documento->id;
            $version->created_at = date('Y-m-d H:i:s');
            $version->updated_at = date('Y-m-d H:i:s');
            
            // Valida si hay archivo adjunto
            if($request->archivo){
                $extension = $request->archivo->getClientOriginalExtension();
                // Crea el nombre, guarda el archivo y guarda la ruta en la BD
                $nombre = $documento->codigo .' - '. $documento->descripcion .' - V'. $version->version .'.'. $extension;
                Storage::disk('private')->put('/documentos/' . $nombre, \File::get($request->archivo));
                $version->url = 'documentos/' . $nombre;
                
                // Define el tipo de archivo
                if ($extension == 'pdf') {
                    $version->tipo = 'pdf';
                    if (!isset($request->descargar)) {
                        $documento->descargar = false;
                        $documento->save();
                    }
                } else if ($extension == 'doc' || $extension == 'docx') {
                    $version->tipo = 'word';
                } else if ($extension == 'xls' || $extension == 'xlsx') {
                    $version->tipo = 'excel';
                } else if ($extension == 'ppt' || $extension == 'pptx') {
                    $version->tipo = 'powerpoint';
                }
            }
            if (!$version->tipo) {
                $version->tipo = 'alt';
            }
            $version->save();
            
            // Se filtra el request para saber cuales son los módulos que están seleccionados
            $cargosSeleccionados = array_filter($request->all(), function($item) {
                return str_contains($item, 'chk-cargo');
            }, ARRAY_FILTER_USE_KEY);
            
            // Se arma un array con los cargos seleccionados para insertarlo en la BD
            $cargosInsertar = [];
            foreach ($cargosSeleccionados as $index => $cargo) {
                array_push($cargosInsertar, [
                    'documento_id' => $documento->id,
                    'cargo_id' => str_replace('chk-cargo-', '', $index)
                ]);
            }

            // Se inserta el array a través del método "insertOrIgnore" que evita duplicidades
            DB::table('cargos_documentos')->insertOrIgnore($cargosInsertar);
            
            // Se filtra el request para saber cuales son las regionales que están seleccionadas
            $regionalesSeleccionadas = array_filter($request->all(), function($item) {
                return str_contains($item, 'regional-');
            }, ARRAY_FILTER_USE_KEY);
            
            // Se arma un array con las regionales seleccionadas para insertarlo en la BD
            $regionalesInsertar = [];
            foreach ($regionalesSeleccionadas as $index => $regional) {
                array_push($regionalesInsertar, [
                    'documento_id' => $documento->id,
                    'regional_id' => str_replace('regional-', '', $index)
                ]);
            }

            // Se inserta el array a través del método "insertOrIgnore" que evita duplicidades
            DB::table('documentos_regionales')->insertOrIgnore($regionalesInsertar);

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Documentos';
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 3;
            $auditoria->documento_id = $documento->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Confirma los cambios realizados en la BD y redirige a la vista del documento
            DB::commit();
            return redirect()->route('documentos.show', ['codigo' => $documento->codigo]);
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            
            // Se crea el mensaje de error
            $titulo = 'Error al crear';
            $mensaje = 'Hubo un error al intentar crear el documento. Por favor intente más tarde.';
            $fecha = Carbon::now();
            Session::put('error_general', compact('titulo', 'mensaje', 'fecha'));
            return back();
        }
    }

    public function show($codigo) {
        $documento = Documento::where('codigo', $codigo)->firstOrFail();
        $validacionRegional = true;
        if (Auth::user()->regional_id) {
            $validacionRegional = DB::table('documentos_regionales')
                ->where('documento_id', $documento->id)
                ->where('regional_id', Auth::user()->regional_id)
                ->first();
        }
        if ($documento->version)
            $firma = Firma::where('version_id', $documento->version->id)->where('user_id', Auth::id())->first();
        else
            $firma = null;

        $validacionCargo = DB::table('cargos_documentos')
            ->where('cargo_id', Auth::user()->cargo->id)
            ->where('documento_id', $documento->id)
            ->first();
        
        if (($validacionCargo && $validacionRegional) || Auth::user()->rol_id < 4) {
            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Documentos';
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 8;
            $auditoria->documento_id = $documento->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();
            return view('documentos.show')->with(compact('documento', 'firma', 'validacionCargo', 'validacionRegional'));
        } else {
            abort(403);
        }
    }

    public function edit($codigo) {
        $documento = Documento::where('codigo', $codigo)->withCount('cargos')->firstOrFail();
        $cargosDocumento = $documento->cargos->toArray();
        $departamentos = Departamento::all()->sortBy('descripcion');
        $cargos = Cargo::all()->sortBy('codigo');
        $categorias = Categoria::all()->sortBy('descripcion');
        $regionales = Regional::all()->sortBy('regional');
        
        if (Auth::user()->rol_id < 4) {
            return view('documentos.edit')->with(compact('departamentos', 'cargos', 'categorias', 'regionales', 'documento', 'cargosDocumento'));
        } else {
            abort(403);
        }
    }

    public function update(Request $request, $codigo) {
        if (Auth::user()->rol_id >= 4)
            abort(403);
        
        $documento = Documento::where('codigo', $codigo)->firstOrFail();
        try {
            // Inicia la tansacción
            DB::beginTransaction();
            
            $documento->fill($request->all());
            $documento->codigo = mb_strtoupper(trim($documento->codigo));
            $documento->descripcion = mb_strtoupper(trim($documento->descripcion));
            $documento->firmar = isset($request->firmar);
            if ($documento->version->tipo == 'pdf')
                $documento->descargar = isset($request->descargar);
            else
                $documento->descargar = true;
            $documento->save();
            
            // Se filtra el request para saber cuales son los módulos que están seleccionados
            $cargosSeleccionados = array_filter($request->all(), function($item) {
                return str_contains($item, 'chk-cargo');
            }, ARRAY_FILTER_USE_KEY);

            // Se comparan los cargos del documento con los cargos seleccionados para quitar los que no están
            foreach ($documento->cargos as $index => $cargo) {
                if (!array_key_exists('chk-cargo-' . $cargo->id, $cargosSeleccionados)) {
                    $documento->cargos()->detach($cargo->id);
                }
            }
            
            // Se arma un array con los cargos seleccionados para insertarlo en la BD
            $cargosInsertar = [];
            foreach ($cargosSeleccionados as $index => $cargo) {
                array_push($cargosInsertar, [
                    'documento_id' => $documento->id,
                    'cargo_id' => str_replace('chk-cargo-', '', $index)
                ]);
            }

            // Se inserta el array a través del método "insertOrIgnore" que evita duplicidades
            DB::table('cargos_documentos')->insertOrIgnore($cargosInsertar);
            
            // Se filtra el request para saber cuales son las regionales que están seleccionadas
            $regionalesSeleccionadas = array_filter($request->all(), function($item) {
                return str_contains($item, 'regional-');
            }, ARRAY_FILTER_USE_KEY);

            // Se comparan las regionales del documento con las regionales seleccionadas para quitar las que no están
            foreach ($documento->regionales as $index => $regional) {
                if (!array_key_exists('regional-' . $regional->id, $regionalesSeleccionadas)) {
                    $documento->regionales()->detach($regional->id);
                }
            }
            
            // Se arma un array con las regionales seleccionadas para insertarlo en la BD
            $regionalesInsertar = [];
            foreach ($regionalesSeleccionadas as $index => $regional) {
                array_push($regionalesInsertar, [
                    'documento_id' => $documento->id,
                    'regional_id' => str_replace('regional-', '', $index)
                ]);
            }

            // Se inserta el array a través del método "insertOrIgnore" que evita duplicidades
            DB::table('documentos_regionales')->insertOrIgnore($regionalesInsertar);

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Documentos';
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 4;
            $auditoria->documento_id = $documento->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Confirma los cambios realizados en la BD y redirige a la vista del grupo
            DB::commit();
            return redirect()->route('documentos.show', ['codigo' => $documento->codigo]);
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);

            // Se crea el mensaje de error
            $titulo = 'Error al actualizar';
            $mensaje = 'Hubo un error al intentar actualizar el documento ' . $documento->codigo . '. Por favor intente más tarde.';
            $fecha = Carbon::now();
            Session::put('error_general', compact('titulo', 'mensaje', 'fecha'));
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function delete($codigo) {
        if (Auth::user()->rol_id >= 4)
            abort(403);
        
        $hoy = Carbon::now();
        $documento = Documento::where('codigo', $codigo)->firstOrFail();
        return view('documentos.delete')->with(compact('documento', 'hoy'));
    }

    public function destroy($codigo) {
        if (Auth::user()->rol_id >= 4)
            abort(403);
        
        $documento = Documento::where('codigo', $codigo)->withCount('cargos')->firstOrFail();
        try {
            // Inicia la tansacción
            DB::beginTransaction();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Documentos';
            $auditoria->observacion = 'Versiones eliminadas: ' . $documento->versiones->count();
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 6;
            $auditoria->documento_id = $documento->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            foreach ($documento->versiones as $i => $version) {
                $version->delete();
            }

            $documento->delete();
            
            // Confirma los cambios realizados en la BD y redirige a la vista del usuario
            DB::commit();
            return redirect()->route('documentos.index');
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            
            // Se crea el mensaje de error
            $titulo = 'Error al eliminar';
            $mensaje = 'Hubo un error al intentar eliminar el documento ' . $documento->codigo . '. Por favor intente más tarde.';
            $fecha = Carbon::now();
            Session::put('error_general', compact('titulo', 'mensaje', 'fecha'));
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function sign(Request $request, $codigo) {
        $documento = Documento::where('codigo', $codigo)->firstOrFail();
        $validacionRegional = true;
        if (Auth::user()->regional_id) {
            $validacionRegional = DB::table('documentos_regionales')
                ->where('documento_id', $documento->id)
                ->where('regional_id', Auth::user()->regional_id)
                ->first();
        }

        $validacionCargo = DB::table('cargos_documentos')
            ->where('cargo_id', Auth::user()->cargo->id)
            ->where('documento_id', $documento->id)
            ->first();
        
        if ($validacionCargo && $validacionRegional) {
            $firma = new Firma();
            $firma->ip = Session::has('ip') ? Session::get('ip') : '';
            $firma->ip_avicampo = \Request::ip();
            $firma->user_id = Auth::id();
            $firma->version_id = $documento->version->id;
            if($request->firma){
                $image_64 = $request->firma;
                $extension = explode('/', explode(':', substr($image_64, 0, strpos($image_64, ';')))[1])[1];
                $replace = substr($image_64, 0, strpos($image_64, ',')+1);
    
                $image = str_replace($replace, '', $image_64);
                $image = str_replace(' ', '+', $image);
                $nombre = Auth::user()->documento.'_'.$documento->codigo.'_v'.$documento->version->version.'.'.$extension;
                Storage::disk('private')->put('/firmas/' . $nombre, base64_decode($image));
                $firma->imagen = 'firmas/' . $nombre;
            }
            $firma->created_at = date('Y-m-d H:i:s');
            $firma->save();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Documentos';
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 9;
            $auditoria->documento_id = $documento->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();
            return redirect()->route('documentos.show', ['codigo' => $documento->codigo]);
        } else {
            abort(403);
        }
    }

    public function version($codigo) {
        $documento = Documento::where('codigo', $codigo)->withCount('cargos')->firstOrFail();
        if (Auth::user()->rol_id < 4) {
            return view('documentos.version')->with(compact('documento'));
        } else {
            abort(403);
        }
    }

    public function destroyVersion(Request $request, $codigo, $id) {
        if (Auth::user()->rol_id >= 4)
            abort(403);
        
        $version = Version::where('id', $id)->firstOrFail();
        if ($version->documento->codigo == $codigo) {
            try {
                // Inicia la tansacción
                DB::beginTransaction();
    
                // Guarda la acción en la tabla de auditorias
                $auditoria = new Auditoria();
                $auditoria->modulo = 'Documentos';
                $auditoria->observacion = 'Version ' . $version->version . ' eliminada.';
                $auditoria->user_id = Auth::id();
                $auditoria->accion_id = 6;
                $auditoria->documento_id = $version->documento->id;
                $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
                $auditoria->ip_avicampo = \Request::ip();
                $auditoria->save();
    
                $version->delete();
                
                // Confirma los cambios realizados en la BD y redirige a la vista del usuario
                DB::commit();
                return response()->json(compact('version'), 200);
            } catch (\Exception $exc) {
                // En caso de fallar reversa los cambios realizados en la BD
                DB::rollback();
                report($exc);
                return response()->withErrors(['msg', 'Hubo un error inesperado']);
            }
        } else {
            return response()->json(compact('version'), 400);
        }
    }
    
    public function save(Request $request, $codigo) {
        $documento = Documento::where('codigo', $codigo)->withCount('cargos')->firstOrFail();
        $version = new Version();
        try {
            // Inicia la tansacción
            DB::beginTransaction();
            
            $version->fill($request->all());
            $version->documento_id = $documento->id;
            $version->created_at = date('Y-m-d H:i:s');
            $version->updated_at = date('Y-m-d H:i:s');
            
            // Valida si hay archivo adjunto
            if($request->archivo){
                $extension = $request->archivo->getClientOriginalExtension();
                // Crea el nombre, guarda el archivo y guarda la ruta en la BD
                $nombre = $documento->codigo .' - '. $documento->descripcion .' - V'. $version->version .'.'. $extension;
                Storage::disk('private')->put('/documentos/' . $nombre, \File::get($request->archivo));
                $version->url = 'documentos/' . $nombre;
                
                // Define el tipo de archivo
                if ($extension == 'pdf') {
                    $version->tipo = 'pdf';
                    if (!isset($request->descargar)) {
                        $documento->descargar = false;
                        $documento->save();
                    }
                } else if ($extension == 'doc' || $extension == 'docx') {
                    $version->tipo = 'word';
                } else if ($extension == 'xls' || $extension == 'xlsx') {
                    $version->tipo = 'excel';
                } else if ($extension == 'ppt' || $extension == 'pptx') {
                    $version->tipo = 'powerpoint';
                }
            }
            if (!$version->tipo) {
                $version->tipo = 'alt';
            }
            $version->save();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Documentos';
            $auditoria->observacion = 'Se sube la versión ' . $version->version . ' en formato ' . $version->tipo .'.';
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 5;
            $auditoria->documento_id = $documento->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Confirma los cambios realizados en la BD y redirige a la vista del grupo
            DB::commit();
            return redirect()->route('documentos.edit', ['codigo' => $documento->codigo]);
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            
            // Se crea el mensaje de error
            $titulo = 'Error al cargar';
            $mensaje = 'Hubo un error al intentar cargar la versión del documento ' . $documento->codigo . '. Por favor intente más tarde.';
            $fecha = Carbon::now();
            Session::put('error_general', compact('titulo', 'mensaje', 'fecha'));
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function download(Request $request) {
        $documento = Documento::findOrFail($request->id);
        $version = $documento->version;

        // Guarda la acción en la tabla de auditorias
        $auditoria = new Auditoria();
        $auditoria->modulo = 'Documentos';
        $auditoria->user_id = Auth::id();
        $auditoria->accion_id = 7;
        $auditoria->documento_id = $documento->id;
        $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
        $auditoria->ip_avicampo = \Request::ip();
        $auditoria->save();

        return response()->json(compact('documento', 'version'), 200);
    }

    public function validation($codigo, $version) {
        $documento = Documento::where('codigo', $codigo)->first();
        if ($documento) {
            $version = $documento->versiones->where('version', $version)->first();
        } else {
            $version = null;
        }
        return response()->json(compact('documento', 'version'), 200);
    }
}
