<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Categoria;
use App\Documento;
use App\Proceso;
use Session;

class DashboardController extends Controller
{
    // Define que vista renderizar y con que información
    public function index() {
        if (!Auth::user()->cargo) {
            $idMenu = 'dashboard';
            $mensaje = 'Hola <b class="capitalize">'. mb_strtolower(Auth::user()->nombre). '</b>, debes tener un cargo asignado antes de poder visualizar documentos.';
            return view('error')->with(compact('idMenu', 'mensaje'));
        }
        $procesos = Proceso::all()->sortBy('codigo');
        $total_documentos = 0;
        foreach ($procesos as $index => $proceso) {
            $proceso->documentos = Documento::whereRaw('SUBSTRING(codigo, 7, 2) = ?', [$proceso->codigo])
                ->when(Auth::user()->regional_id, function ($query) {
                    return  $query->join('documentos_regionales AS dcr', function ($join) {
                        $join->on('documentos.id', 'dcr.documento_id')
                        ->where('dcr.regional_id', Auth::user()->regional_id);
                    });
                })
                ->join('cargos_documentos AS cxd', function ($join) {
                    $join->on('documentos.id', 'cxd.documento_id')
                    ->where('cxd.cargo_id', Auth::user()->cargo->id);
                })
                ->join('versiones AS ver', function ($join) {
                    $join->on('documentos.id', 'ver.documento_id')
                    ->where('ver.version', DB::raw("(SELECT MAX(`version`) FROM versiones WHERE documento_id = ver.documento_id AND deleted_at IS NULL)"));
                })
                ->count();
                $total_documentos = $total_documentos + $proceso->documentos;
        }
        return view('dashboard.user')->with(compact('procesos', 'total_documentos'));
    }

    // Lista los documentos pertenecientes a algún proceso
    public function proceso($proceso) {
        if ($proceso != 'todos') {
            $proceso = Proceso::where('slug', $proceso)->firstOrFail();
        }
        
        $orden = (isset($_GET['orden']) ? substr($_GET['orden'], 0, strpos($_GET['orden'], '|')) : 'doc.created_at');
        $sort = (isset($_GET['orden']) ? substr($_GET['orden'], strpos($_GET['orden'], '|') + 1) : 'desc');
        $tipo = (isset($_GET['tipo']) ? $_GET['tipo'] : null);
        $nombre = (isset($_GET['nombre']) ? $_GET['nombre'] : null);

        if (isset($_GET['perPage'])) {
            Session::put('pagination', $_GET['perPage']);
        } elseif (!Session::has('pagination')) {
            Session::put('pagination', 10);
        }

        $cargo = Auth::user()->cargo->id;
        $regional = Auth::user()->regional_id;
        $categorias = Categoria::all()->sortBy('descripcion');
        $documentos = DB::table('documentos AS doc')
            ->leftJoin('categorias AS cat', 'doc.categoria_id', 'cat.id')
            ->when($regional, function ($query) use ($regional) {
                return  $query->join('documentos_regionales AS dcr', function ($join) use ($regional) {
                    $join->on('doc.id', 'dcr.documento_id')
                    ->where('dcr.regional_id', $regional);
                });
            })
            ->join('cargos_documentos AS cxd', function ($join) use ($cargo) {
                $join->on('doc.id', 'cxd.documento_id')
                ->where('cxd.cargo_id', $cargo);
            })
            ->join('versiones AS ver', function ($join) {
                $join->on('doc.id', 'ver.documento_id')
                ->where('ver.version', DB::raw("(SELECT MAX(`version`) FROM versiones WHERE documento_id = ver.documento_id AND deleted_at IS NULL)"));
            })
            ->leftJoin('firmas AS fir', function ($join) {
                $join->on('ver.id', 'fir.version_id')
                ->where('fir.user_id', Auth::user()->id);
            })
            ->select('doc.id', 'doc.codigo', 'doc.descripcion', 'doc.firmar', 'ver.version', 'ver.tipo', 'ver.url', 'fir.id AS firma', 'cat.descripcion AS categoria', 'ver.created_at', 'doc.updated_at')
            ->whereNull('doc.deleted_at')
            ->when($proceso != 'todos', function ($query) use ($proceso) {
                return $query->whereRaw('SUBSTRING(codigo, 7, 2) = ?', [$proceso->codigo]);
            })
            ->when($tipo, function ($query) use ($tipo) {
                return $query->where('doc.categoria_id', $tipo);
            })
            ->when($nombre, function ($query) use ($nombre) {
                return $query->whereRaw('UPPER(doc.descripcion) LIKE ?', '%' . mb_strtoupper(str_replace(' ', '%', $nombre)) . '%')
                            ->orWhereRaw('UPPER(doc.codigo) LIKE ?', '%' . mb_strtoupper(str_replace(' ', '%', $nombre)) . '%');
            })
            ->orderBy($orden, $sort)
            ->paginate(Session::get('pagination'));
        return view('dashboard.procesos')->with(compact('proceso', 'documentos', 'categorias', 'orden', 'sort', 'tipo', 'nombre'));
    }

    public function ip(Request $request) {
        Session::put('ip', $request->ip);
        $ip = $request->ip;
        return response()->json(compact('ip'), 200);
    }
}
