<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Accion;
use App\Area;
use App\Auditoria;
use App\Cargo;
use App\Departamento;
use App\Regional;
use App\Rol;
use App\User;
use Carbon\Carbon;
use Session;

class CargosController extends Controller
{
    public function index() {
        if (Auth::user()->rol_id < 3) {
            $hoy = Carbon::now();
            $departamento = (isset($_GET['departamento']) ? $_GET['departamento'] : null);
            $area = ($departamento && isset($_GET['area']) ? $_GET['area'] : null);
            $descripcion = (isset($_GET['descripcion']) ? $_GET['descripcion'] : null);

            if (isset($_GET['perPage'])) {
                Session::put('pagination', $_GET['perPage']);
            } elseif (!Session::has('pagination')) {
                Session::put('pagination', 10);
            }
            
            $departamentos = Departamento::all()->sortBy('descripcion');
            $areas = Area::all()->sortBy('descripcion');
            $cargos = Cargo::join('areas AS are', 'cargos.area_id', 'are.id')
                ->join('departamentos AS dep', 'are.departamento_id', 'dep.id')
                ->when($departamento, function ($query) use ($departamento) {
                    return $query->where('dep.id', $departamento);
                })
                ->when($area, function ($query) use ($area) {
                    return $query->where('are.id', $area);
                })
                ->when($descripcion, function ($query) use ($descripcion) {
                    return $query->whereRaw('UPPER(cargos.descripcion) LIKE ?', '%' . mb_strtoupper(str_replace(' ', '%', $descripcion)) . '%');
                })
                ->select('cargos.*', 'are.descripcion as area_desc', 'dep.descripcion as departamento_desc')
                ->orderBy('departamento_desc', 'asc')
                ->orderBy('area_desc', 'asc')
                ->orderBy('cargos.descripcion', 'asc')
                ->paginate(Session::get('pagination'));
            return view('cargos.index')->with(compact('cargos', 'departamentos', 'areas', 'departamento', 'area', 'descripcion', 'hoy'));
        } else {
            abort(403);
        }
    }

    public function create() {
        if (Auth::user()->rol_id < 3) {
            $departamentos = Departamento::all()->sortBy('descripcion');
            $areas = Area::all()->sortBy('descripcion');
            $cargos = Cargo::all()->sortBy('descripcion');
            return view('cargos.data')->with(compact('departamentos', 'areas', 'cargos'));
        } else {
            abort(403);
        }
    }
    
    public function store(Request $request) {
        if (Auth::user()->rol_id >= 3)
            abort(403);
        
        $usuario = new User();
        try {
            // Inicia la tansacción
            DB::beginTransaction();

            $usuario->fill($request->all());
            $usuario->password = Hash::make($request->documento);
            if ($usuario->regional_id == 'null');
                $usuario->regional_id = null;
            $usuario->save();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Usuarios';
            $auditoria->observacion = 'Usuario creado: ' . $usuario->nombre . ' con número de documento ' . $usuario->documento;
            $auditoria->user_id = Auth::id();
            $auditoria->usuario_id = $usuario->id;
            $auditoria->accion_id = 5;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Confirma los cambios realizados en la BD y redirige a la vista del usuario
            DB::commit();
            return redirect()->route('usuarios.ver', ['documento' => $usuario->documento]);
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            dd($exc);
            $success = false;
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function show($cargo) {
        if (Auth::user()->rol_id < 3) {
            $hoy = Carbon::now();
            $cargo = Cargo::findOrFail($cargo);
            
            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Cargos';
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 8;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();
            return view('cargos.ver')->with(compact('cargo', 'hoy'));
        } else {
            abort(403);
        }
    }

    public function traceability($documento) {
        if (Auth::user()->rol_id < 3) {
            $hoy = Carbon::now();
            $usuario = User::where('documento', $documento)->firstOrFail();

            $fecha_inicial = (isset($_GET['fecha_inicial']) ? $_GET['fecha_inicial'] : Carbon::now()->firstOfMonth()->startOfDay()->toDateString());
            $fecha_final = (isset($_GET['fecha_final']) ? $_GET['fecha_final'] : $hoy->endOfDay()->toDateString());
            $accion = (isset($_GET['accion']) ? $_GET['accion'] : null);

            if (isset($_GET['perPage'])) {
                Session::put('pagination', $_GET['perPage']);
            } elseif (!Session::has('pagination')) {
                Session::put('pagination', 10);
            }

            $acciones = Accion::all()->sortBy('descripcion');
            $auditorias = Auditoria::where('user_id', $usuario->id)
                ->whereDate('created_at', '>=', $fecha_inicial)
                ->whereDate('created_at', '<=', $fecha_final)
                ->when($accion, function ($query) use ($accion) {
                    return $query->where('accion_id', $accion);
                })
                ->orderBy('created_at', 'desc')
                ->paginate(Session::get('pagination'));
            return view('usuarios.traceability')->with(compact('usuario', 'hoy', 'auditorias', 'acciones', 'fecha_inicial', 'fecha_final', 'accion'));
        } else {
            abort(403);
        }
    }

    public function edit($documento) {
        if (Auth::user()->rol_id < 3) {
            $usuario = User::where('documento', $documento)->firstOrFail();
            $departamentos = Departamento::all()->sortBy('descripcion');
            $areas = Area::all()->sortBy('descripcion');
            $cargos = Cargo::all()->sortBy('descripcion');
            $regionales = Regional::all()->sortBy('nombre');
            $roles = Rol::all()->sortByDesc('id');
            return view('usuarios.data')->with(compact('usuario', 'departamentos', 'regionales', 'areas', 'cargos', 'roles'));
        } else {
            abort(403);
        }
    }

    public function update(Request $request, $documento) {
        if (Auth::user()->rol_id >= 3)
            abort(403);
        
        $usuario = User::where('documento', $documento)->firstOrFail();
        try {
            // Inicia la tansacción
            DB::beginTransaction();

            $usuario->fill($request->all());
            $usuario->password = Hash::make($request->documento);
            if ($usuario->regional_id == 'null');
                $usuario->regional_id = null;
            $usuario->save();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Usuarios';
            $auditoria->observacion = 'Usuario modificado: ' . $usuario->nombre . ' con número de documento ' . $usuario->documento;
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 4;
            $auditoria->usuario_id = $usuario->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Confirma los cambios realizados en la BD y redirige a la vista del usuario
            DB::commit();
            return redirect()->route('usuarios.ver', ['documento' => $usuario->documento]);
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            dd($exc);
            $success = false;
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function delete($documento) {
        $hoy = Carbon::now();
        $usuario = User::where('documento', $documento)->firstOrFail();
        if (Auth::user()->rol_id < 3 && Auth::user()->rol_id < $usuario->rol_id) {
            return view('usuarios.eliminar')->with(compact('usuario', 'hoy'));
        } else {
            abort(403);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $documento)
    {
        $usuario = User::where('documento', $documento)->firstOrFail();
        if (Auth::user()->rol_id >= 3 || Auth::user()->rol_id >= $usuario->rol_id)
            abort(403);
        
        try {
            // Inicia la tansacción
            DB::beginTransaction();

            // Guarda la acción en la tabla de auditorias
            $auditoria = new Auditoria();
            $auditoria->modulo = 'Usuarios';
            $auditoria->observacion = 'Usuario eliminado: ' . $usuario->nombre . ' con número de documento ' . $usuario->documento;
            $auditoria->user_id = Auth::id();
            $auditoria->accion_id = 6;
            $auditoria->usuario_id = $usuario->id;
            $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
            $auditoria->ip_avicampo = \Request::ip();
            $auditoria->save();

            // Borra al usuario
            $usuario->delete();
            
            // Confirma los cambios realizados en la BD y redirige a la vista del usuario
            DB::commit();
            return redirect()->route('usuarios');
        } catch (\Exception $exc) {
            // En caso de fallar reversa los cambios realizados en la BD
            DB::rollback();
            report($exc);
            dd($exc);
            $success = false;
            return back()->withErrors(['msg', 'Hubo un error inesperado']);
        }
    }

    public function profile() {
        $hoy = Carbon::now();
        $usuario = Auth::user();

        // Guarda la acción en la tabla de auditorias
        $auditoria = new Auditoria();
        $auditoria->modulo = 'Usuarios';
        $auditoria->user_id = Auth::id();
        $auditoria->accion_id = 8;
        $auditoria->usuario_id = $usuario->id;
        $auditoria->ip = Session::has('ip') ? Session::get('ip') : '';
        $auditoria->ip_avicampo = \Request::ip();
        $auditoria->save();
        return view('usuarios.ver')->with(compact('usuario', 'hoy'));
    }

    public function validation($documento) {
        if (Auth::user()->rol_id < 3) {
            $usuario = User::where('documento', $documento)->first();
            return response()->json(compact('usuario'), 200);
        } else {
            abort(403);
        }
    }
}
